/* Copyright (C) 2014-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "EventHandles.h"
#include <cassert>
#include <cstring>
#include <new>

using namespace vncdiscovery;

#define INITIAL_EVENT_HANDLE_SIZE 16

#ifndef NDEBUG
// Class used for checking the invariance of the DeviceProvider
template <typename T>
class InvarianceChecker
{
public:
  InvarianceChecker(const T& obj)
    : mObj(obj)
  {
    mObj.checkInvariance();
  }
  ~InvarianceChecker()
  {
    mObj.checkInvariance();
  }
private:
  const T& mObj;
};

#define __CHECK_INVARIANT \
  InvarianceChecker<EventHandles::HandleSet> __invarianceChecker(*this)
#define __CHECK_INVARIANT_ONCE checkInvariance()
#else
#define __CHECK_INVARIANT
#define __CHECK_INVARIANT_ONCE
#endif

EventHandles::EventHandles()
{
}

EventHandles::~EventHandles()
{
}

void EventHandles::reset()
{
  vnccommon::Mutex::Locker lock(mMutex);
  mModifiableHandles.reset();
}

void EventHandles::addHandle(VNCDiscovererEventHandle *handleToAdd,
    bool writeNotification)
{
  vnccommon::Mutex::Locker lock(mMutex);
  mModifiableHandles.addHandle(handleToAdd, writeNotification);
}

void EventHandles::removeHandle(VNCDiscovererEventHandle *handleToRemove)
{
  vnccommon::Mutex::Locker lock(mMutex);
  mModifiableHandles.removeHandle(handleToRemove);
}


void EventHandles::setHandleWriteNotifications(VNCDiscovererEventHandle *handle,
    bool writeNotification)
{
  vnccommon::Mutex::Locker lock(mMutex);
  mModifiableHandles.setHandleWriteNotifications(handle, writeNotification);
}

bool EventHandles::update()
{
  vnccommon::Mutex::Locker lock(mMutex);
  if(mSdkHandles != mModifiableHandles)
  {
    mSdkHandles = mModifiableHandles;
    return true;
  }
  return false;
}

VNCDiscovererEventHandle* EventHandles::handles()
{
  // No need to lock as the SDK handles are supposed to be always written and
  // accessed in the SDK thread.
  return mSdkHandles.handles();
}

int* EventHandles::writeNotifications()
{
  // No need to lock as the SDK handles are supposed to be always written and
  // accessed in the SDK thread.
  return mSdkHandles.writeNotifications();
}

EventHandles::HandleSet::HandleSet()
  : mpEventHandles(NULL),
    mpWriteNotification(NULL),
    mHandleCount(0),
    mMaxHandleCount(0)
{
  __CHECK_INVARIANT_ONCE;
}

EventHandles::HandleSet::~HandleSet()
{
  __CHECK_INVARIANT_ONCE;
  delete[] mpEventHandles;
  delete[] mpWriteNotification;
}

void EventHandles::HandleSet::reset()
{
  __CHECK_INVARIANT;
  if(mHandleCount > 0)
  {
    mHandleCount = 0;
    mpEventHandles[0] = VNCDiscovererNoEventHandle;
  }
}

void EventHandles::HandleSet::addHandle(VNCDiscovererEventHandle *handleToAdd,
    bool writeNotification)
{
  assert(handleToAdd);
  assert(*handleToAdd != VNCDiscovererNoEventHandle);
  __CHECK_INVARIANT;
  if(!handleToAdd || *handleToAdd == VNCDiscovererNoEventHandle)
  {
    // Nothing to copy
    return;
  }

  for(size_t i = 0; i < mHandleCount; ++i)
  {
    if(*handleToAdd == mpEventHandles[i])
    {
      // Already in the list
      return;
    }
  }

  VNCDiscovererEventHandle*     pTmpEventHandles = NULL;
  int*                          pTmpWriteNotification = NULL;
  try
  {
    if(!mpEventHandles)
    {
      // Create a new array, and set the mMaxHandleCount, taking care to leave
      // enough space for a terminating VNCDiscovererNoEventHandle element.
      assert(mHandleCount == 0);
      pTmpEventHandles = new VNCDiscovererEventHandle[INITIAL_EVENT_HANDLE_SIZE];
      pTmpWriteNotification = new int[INITIAL_EVENT_HANDLE_SIZE];
      mpEventHandles = pTmpEventHandles;
      pTmpEventHandles = NULL;
      mpWriteNotification = pTmpWriteNotification;
      pTmpWriteNotification = NULL;
      mMaxHandleCount = INITIAL_EVENT_HANDLE_SIZE-1;
      mpEventHandles[0] = VNCDiscovererNoEventHandle;
    }
    else if(mHandleCount == mMaxHandleCount)
    {
      assert(mMaxHandleCount > 0);
      // Allocate a new array and copy the contents of the current one to the
      // new one. Since the array is VNCDiscovererNoEventHandle-terminated, add
      // 1 to the maxHandleCount.
      pTmpEventHandles = new VNCDiscovererEventHandle[mMaxHandleCount*2+1];
      pTmpWriteNotification = new int[mMaxHandleCount*2+1];

      // Copy the arrays across and delete the old ones.
      memcpy(pTmpEventHandles, mpEventHandles,
          sizeof(VNCDiscovererEventHandle)*(mHandleCount+1));
      assert(pTmpEventHandles[mHandleCount-1] == mpEventHandles[mHandleCount-1]);
      assert(pTmpEventHandles[mHandleCount] == VNCDiscovererNoEventHandle);
      memcpy(pTmpWriteNotification, mpWriteNotification,
          sizeof(VNCDiscovererEventHandle)*(mHandleCount+1));
      mMaxHandleCount *= 2;
      delete[] mpEventHandles;
      delete[] mpWriteNotification;
      mpEventHandles = pTmpEventHandles;
      pTmpEventHandles = NULL;
      mpWriteNotification = pTmpWriteNotification;
      pTmpWriteNotification = NULL;
    }
  }
  catch(const std::bad_alloc&)
  {
    if(pTmpEventHandles)
    {
      delete[] pTmpEventHandles;
    }
    if(pTmpWriteNotification)
    {
      delete[] pTmpWriteNotification;
    }
    throw;
  }

  assert(mHandleCount < mMaxHandleCount);
  mpEventHandles[mHandleCount] = *handleToAdd;
  mpWriteNotification[mHandleCount] = (writeNotification ? 1 : 0);
  ++mHandleCount;
  mpEventHandles[mHandleCount] = VNCDiscovererNoEventHandle;
}

void EventHandles::HandleSet::setHandleWriteNotifications(
    VNCDiscovererEventHandle *handle, bool writeNotification)
{
  assert(handle);
  assert(*handle != VNCDiscovererNoEventHandle);
  __CHECK_INVARIANT;
  if(!handle || *handle == VNCDiscovererNoEventHandle)
  {
    // Nothing to do
    return;
  }

  // Find the handle in the list
  for(size_t i = 0; i < mHandleCount; ++i)
  {
    if(*handle == mpEventHandles[i])
    {
      // Update the write notification state to the requested value
      mpWriteNotification[i] = (writeNotification ? 1 : 0);
      return;
    }
  }
  // Silently ignore failing to find the handle.
}

void EventHandles::HandleSet::removeHandle(VNCDiscovererEventHandle *handleToRemove)
{
  __CHECK_INVARIANT;
  if(!handleToRemove || *handleToRemove == VNCDiscovererNoEventHandle)
  {
    // Nothing to copy
    return;
  }

  for(size_t i = 0; i < mHandleCount; ++i)
  {
    if(*handleToRemove == mpEventHandles[i])
    {
      // Just move all the remaining handles left by one.
      // The terminating VNCDiscovererNoEventHandle is moved as well.
      for(size_t j = i; j < mHandleCount; ++j)
      {
        mpEventHandles[j] = mpEventHandles[j+1];
        mpWriteNotification[j] = mpWriteNotification[j+1];
      }
      --mHandleCount;
      return;
    }
  }
}

VNCDiscovererEventHandle* EventHandles::HandleSet::handles()
{
  __CHECK_INVARIANT_ONCE;
  return mpEventHandles;
}

int* EventHandles::HandleSet::writeNotifications()
{
  __CHECK_INVARIANT_ONCE;
  return mpWriteNotification;
}

bool EventHandles::HandleSet::operator==(const EventHandles::HandleSet& other) const
{
  __CHECK_INVARIANT_ONCE;
  other.checkInvariance();
  // First check the number of elements. If they have same number, and the
  // number is greater than 0, check each element.
  if(mHandleCount != other.mHandleCount)
  {
    return false;
  }
  if(mHandleCount != 0)
  {
    for(size_t i = 0; i < mHandleCount; ++i)
    {
      if(mpEventHandles[i] != other.mpEventHandles[i] ||
          mpWriteNotification[i] != other.mpWriteNotification[i])
      {
        return false;
      }
    }
  }
  return true;
}

bool EventHandles::HandleSet::operator!=(const EventHandles::HandleSet& other) const
{
  return !(*this == other);
}

EventHandles::HandleSet& EventHandles::HandleSet::operator=(
    const EventHandles::HandleSet& other)
{
  __CHECK_INVARIANT;
  other.checkInvariance();
  reset();
  // First increase the max size of the array if needed.
  if(mMaxHandleCount < other.mHandleCount)
  {
    VNCDiscovererEventHandle*     pTmpEventHandles = NULL;
    int*                          pTmpWriteNotification = NULL;
    try
    {
      pTmpEventHandles = new VNCDiscovererEventHandle[other.mHandleCount+1];
      pTmpWriteNotification = new int[other.mHandleCount+1];
      mMaxHandleCount = other.mHandleCount;
      if(mpEventHandles)
      {
        delete[] mpEventHandles;
      }
      mpEventHandles = pTmpEventHandles;
      if(mpWriteNotification)
      {
        delete[] mpWriteNotification;
      }
      mpWriteNotification = pTmpWriteNotification;
    }
    catch(std::bad_alloc&)
    {
      if(pTmpEventHandles)
      {
        delete[] pTmpEventHandles;
      }
      if(pTmpWriteNotification)
      {
        delete[] pTmpWriteNotification;
      }
      throw;
    }
  }

  mHandleCount = other.mHandleCount;
  // If there are handles defined, then copy them. Otherwise set the first
  // handle to mark the end of the array, bearing in mind the array may be
  // NULL.
  if(mHandleCount > 0)
  {
    for(size_t i = 0; i < mHandleCount+1; ++i)
    {
      mpEventHandles[i] = other.mpEventHandles[i];
      mpWriteNotification[i] = other.mpWriteNotification[i];
    }
  }
  else if(mpEventHandles)
  {
    mpEventHandles[0] = VNCDiscovererNoEventHandle;
  }
  return *this;
}

void EventHandles::HandleSet::checkInvariance() const
{
#ifndef NDEBUG
  assert(mpEventHandles || (mMaxHandleCount == 0 && !mpWriteNotification));
  assert(!mpEventHandles || (mMaxHandleCount > 0 && mpWriteNotification));
  assert(mHandleCount <= mMaxHandleCount);
  assert(mHandleCount == 0 || (mpEventHandles && mpWriteNotification));
  assert(mMaxHandleCount == 0 || (mpEventHandles && mpWriteNotification));
  for(size_t i = 0; i < mHandleCount; ++i)
  {
    assert(mpEventHandles[i] != VNCDiscovererNoEventHandle);
    for(size_t j = i+1; j < mHandleCount; ++j)
    {
      assert(mpEventHandles[i] != mpEventHandles[j]);
    }
  }
  assert(mMaxHandleCount == 0 || 
      mpEventHandles[mHandleCount] == VNCDiscovererNoEventHandle);
#endif
}

